package cn.com.bril.androidocr.studio.engine;

import android.app.AlertDialog;
import android.app.AlertDialog.Builder;
import android.content.Context;
import android.content.DialogInterface;
import android.content.res.TypedArray;
import android.preference.ListPreference;
import android.util.AttributeSet;
import android.widget.ListView;
import android.widget.Toast;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import cn.com.bril.androidocr.studio.R;


public class ListPreferenceMultiSelect extends ListPreference {

	public static interface Filter {
		public boolean isEntryVisible(CharSequence entry);
	}

	private static final String DEFAULT__SEPARATOR = ";";

	private String _separator;
	private boolean[] _clickedEntryIndices;
	private Filter _filter;
	private boolean[] _filterMask;
	private int _filteredEntriesArrayLength;

	private int _checkedEntriesCount = 0;

	private int _minAvailableCheckedEntriesCount = 0;
	private String _errorMinEntriesCountReached;
	private String _errorMinEntriesCountExceeded;

	private int _maxAvailableCheckedEntriesCount = Integer.MAX_VALUE;
	private String _errorMaxEntriesCountReached;
	private String _errorMaxEntriesCountExceeded;

	public ListPreferenceMultiSelect(final Context context, final AttributeSet attrs ) {
		super( context, attrs );

		final TypedArray attributes =
				context.obtainStyledAttributes( attrs, R.styleable.ListPreferenceMultiSelect );
		_separator = attributes.getString( R.styleable.ListPreferenceMultiSelect_separator );
		if( _separator == null ) {
			_separator = ListPreferenceMultiSelect.DEFAULT__SEPARATOR;
		}
		_clickedEntryIndices = new boolean[getEntries().length];
		attributes.recycle();
	}

	public void setFilter( final Filter filter ) {
		_filter = filter;
	}

	/**
	 * Set constraint for checked entries count.
	 * This constraint is guaranteed to be not violated only during an interaction with the user.
	 * So, this constraint can be violated because
	 * 1) checked entries count doesn't fit to constraint bounds after view initialization;
	 * 2) checked entries count is less then {@code minAvailableCheckedEntriesCount} after entries filtration;
	 * When constraint is violated the warning message is shown.
	 * 
	 * @param minAvailableCheckedEntriesCount
	 *            Minimal checked entries count constraint.
	 * @param errorMinEntriesCountReached
	 *            Message to be shown when minimal entries count is reached.
	 *            Can be null. If null then no message is shown.
	 * @param errorMinEntriesCountExceeded
	 *            Message to be shown when minimal entries count is exceeded.
	 *            Can be null. If null then no message is shown.
	 * @param maxAvailableCheckedEntriesCount
	 *            Maximum checked entries count constraint.
	 * @param errorMaxEntriesCountReached
	 *            Message to be shown when maximum entries count is reached.
	 *            Can be null. If null then no message is shown.
	 * @param errorMaxEntriesCountExceeded
	 *            Message to be shown when maximum entries count is exceeded.
	 *            Can be null. If null then no message is shown.
	 * @throws IllegalArgumentException
	 *             (minAvailableCheckedEntriesCount < 0) or (maxAvailableCheckedEntriesCount < 0) or
	 *             (minAvailableCheckedEntriesCount > maxAvailableCheckedEntriesCount)
	 */
	public void setEntriesCountConstraint( final int minAvailableCheckedEntriesCount,
			final String errorMinEntriesCountReached, final String errorMinEntriesCountExceeded,
			final int maxAvailableCheckedEntriesCount, final String errorMaxEntriesCountReached,
			final String errorMaxEntriesCountExceeded ) throws IllegalArgumentException {
		if( minAvailableCheckedEntriesCount < 0 ) {
			throw new IllegalArgumentException(
					"minAvailableCheckedEntriesCount should be greater or equal then zero." );
		}
		if( maxAvailableCheckedEntriesCount < 0 ) {
			throw new IllegalArgumentException(
					"maxAvailableCheckedEntriesCount should be greater or equal then zero." );
		}
		if( minAvailableCheckedEntriesCount > maxAvailableCheckedEntriesCount ) {
			throw new IllegalArgumentException(
					"minAvailableCheckedEntriesCount should be less or equal then maxAvailableCheckedEntriesCount." );
		}

		_minAvailableCheckedEntriesCount = minAvailableCheckedEntriesCount;
		_errorMinEntriesCountReached = errorMinEntriesCountReached;
		_errorMinEntriesCountExceeded = errorMinEntriesCountExceeded;

		_maxAvailableCheckedEntriesCount = maxAvailableCheckedEntriesCount;
		_errorMaxEntriesCountReached = errorMaxEntriesCountReached;
		_errorMaxEntriesCountExceeded = errorMaxEntriesCountExceeded;
	}

	@Override
	public CharSequence[] getEntries() {
		CharSequence[] entries = super.getEntries();
		if( _filter != null ) {
			// Allocate array for filter mask
			_filterMask = new boolean[entries.length];
			// Filter entries and store filter mask for getEntriesValues()
			final ArrayList<CharSequence> filteredEntries = new ArrayList<CharSequence>();
			int i = 0;
			for( final CharSequence entry : entries ) {
				final boolean isEntryVisible = _filter.isEntryVisible( entry );
				_filterMask[i++] = isEntryVisible;
				if( isEntryVisible ) {
					filteredEntries.add( entry );
				}
			}
			_filteredEntriesArrayLength = filteredEntries.size();
			entries = new CharSequence[_filteredEntriesArrayLength];
			filteredEntries.toArray( entries );
		}
		return entries;
	}

	@Override
	public CharSequence[] getEntryValues() {
		CharSequence[] entryValues = super.getEntryValues();
		if( _filterMask != null ) {
			final CharSequence[] filteredEntryValues = new CharSequence[_filteredEntriesArrayLength];
			int j = 0;
			for( int i = 0; i < entryValues.length; ++i ) {
				if( _filterMask[i] ) {
					filteredEntryValues[j++] = entryValues[i];
				}
			}
			entryValues = filteredEntryValues;
		}
		return entryValues;
	}

	@Override
	public void setEntries( final CharSequence[] entries ) {
		super.setEntries( entries );
		_clickedEntryIndices = new boolean[entries.length];
	}

	public ListPreferenceMultiSelect(final Context context ) {
		this( context, null );
	}

	@Override
	protected void onPrepareDialogBuilder( final Builder builder ) {
		final CharSequence[] entries = getEntries();
		final CharSequence[] entryValues = getEntryValues();
		if( entries == null || entryValues == null || entries.length != entryValues.length ) {
			throw new IllegalStateException(
					"ListPreference requires an entries array and an entryValues array which are both the same length" );
		}

		setCheckedEntries();
		builder.setMultiChoiceItems( entries, _clickedEntryIndices,
				new DialogInterface.OnMultiChoiceClickListener() {
					@Override
					public void onClick( final DialogInterface dialog, final int which, final boolean val ) {
						if( val ) {
							if( _checkedEntriesCount < _maxAvailableCheckedEntriesCount ) {
								++_checkedEntriesCount;
								_clickedEntryIndices[which] = true;

								if( _checkedEntriesCount < _minAvailableCheckedEntriesCount ) {
									if( _errorMinEntriesCountExceeded != null ) {
										final Context context = ( (AlertDialog) dialog ).getContext();
										Toast.makeText( context, _errorMinEntriesCountExceeded, Toast.LENGTH_LONG ).show();
									}
								}
							} else {
								// Maximum checked entries count constraint is violated.
								// The list entry is already checked and we need to uncheck it back.
								final Context context = ( (AlertDialog) dialog ).getContext();
								final ListView listView = ( (AlertDialog) dialog ).getListView();
								_clickedEntryIndices[which] = false;
								listView.setItemChecked( which, false );
								if( _errorMaxEntriesCountReached != null ) {
									Toast.makeText( context, _errorMaxEntriesCountReached, Toast.LENGTH_SHORT ).show();
								}
							}
						} else {
							if( _checkedEntriesCount > _minAvailableCheckedEntriesCount ) {
								--_checkedEntriesCount;
								_clickedEntryIndices[which] = false;

								if( _checkedEntriesCount > _maxAvailableCheckedEntriesCount ) {
									if( _errorMaxEntriesCountExceeded != null ) {
										final Context context = ( (AlertDialog) dialog ).getContext();
										Toast.makeText( context, _errorMaxEntriesCountExceeded, Toast.LENGTH_LONG ).show();
									}
								}
							} else {
								// Minimal checked entries count constraint is violated.
								// The list entry is already unchecked and we need to check it back.
								final Context context = ( (AlertDialog) dialog ).getContext();
								final ListView listView = ( (AlertDialog) dialog ).getListView();
								_clickedEntryIndices[which] = true;
								listView.setItemChecked( which, true );
								if( _errorMinEntriesCountReached != null ) {
									Toast.makeText( context, _errorMinEntriesCountReached, Toast.LENGTH_SHORT ).show();
								}
							}
						}
					}
				} );
	}

	private void setCheckedEntries() {
		final CharSequence[] entryValues = getEntryValues();
//		CharSequence value = getValue();
		//value重定义
		CharSequence value="English;ChineseSimplified";
		String[] values = null;
		if( value.length() > 0 ) {
			values = ( ( String )value ).split( _separator );
		}
		if( values != null ) {
			final List<String> valuesList = Arrays.asList( values );
			_checkedEntriesCount = 0;
			for( int i = 0; i < entryValues.length; i++ ) {
				final CharSequence entry = entryValues[i];
				if( valuesList.contains( entry ) ) {
					_clickedEntryIndices[i] = true;
					++_checkedEntriesCount;
				}
			}
			if( _checkedEntriesCount < _minAvailableCheckedEntriesCount ) {
				if( _errorMinEntriesCountExceeded != null ) {
					final Context context = getContext();
					Toast.makeText( context, _errorMinEntriesCountExceeded, Toast.LENGTH_LONG );
				}
			} else if( _checkedEntriesCount > _maxAvailableCheckedEntriesCount ) {
				if( _errorMaxEntriesCountExceeded != null ) {
					final Context context = getContext();
					Toast.makeText( context, _errorMaxEntriesCountExceeded, Toast.LENGTH_LONG );
				}
			}
		}
	}

	@Override
	protected void onDialogClosed( final boolean result ) {
		final ArrayList<String> values = new ArrayList<String>();
		final CharSequence[] entryValues = getEntryValues();
		if( result && entryValues != null ) {
			for( int i = 0; i < entryValues.length; i++ ) {
				if( _clickedEntryIndices[i] ) {
					final String entryValue = ( String )entryValues[i];
					values.add( entryValue );
				}
			}
			if( callChangeListener( values ) ) {
				setValue( StringUtils.join( values, _separator ) );
			}
		}
	}

	public static boolean contains( final String str, final String rawStr, String separator ) {
		if( separator == null ) {
			separator = ListPreferenceMultiSelect.DEFAULT__SEPARATOR;
		}
		final String[] values = rawStr.split( separator );
		for( final String value : values ) {
			if( value.equals( str ) ) {
				return true;
			}
		}
		return false;
	}
}
