package cn.com.bril.androidocr.studio.ui.pulltorefresh;

import android.annotation.TargetApi;
import android.content.Context;
import android.os.Build.VERSION;
import android.os.Build.VERSION_CODES;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewConfiguration;
import android.view.ViewGroup;
import android.view.ViewTreeObserver.OnGlobalLayoutListener;
import android.view.animation.DecelerateInterpolator;
import android.view.animation.Interpolator;
import android.widget.FrameLayout;
import android.widget.LinearLayout;


public abstract class PullToRefreshBase<T extends View> extends LinearLayout implements IPullToRefresh<T> {

	public interface OnRefreshListener<V extends View> {

		void onPullDownToRefresh(final PullToRefreshBase<V> refreshView);

		void onPullUpToRefresh(final PullToRefreshBase<V> refreshView);
	}

	private static final int SCROLL_DURATION = 200;
	private static final float OFFSET_RADIO = 2.5f;
	private float mLastMotionY = -1;
	private OnRefreshListener<T> mRefreshListener;
	private LoadingLayout mHeaderLayout;
	private LoadingLayout mFooterLayout;
	private int mHeaderHeight;
	private int mFooterHeight;
	private Interpolator mScrollAnimationInterpolator;
	private boolean mPullRefreshEnabled = true;
	private boolean mPullLoadEnabled = false;
	private boolean mScrollLoadEnabled = false;
	private boolean mInterceptEventEnable = true;
	private boolean mIsHandledTouchEvent = false;
	private int mTouchSlop;
	private ILoadingLayout.State mPullDownState = ILoadingLayout.State.NONE;
	private ILoadingLayout.State mPullUpState = ILoadingLayout.State.NONE;
	T mRefreshableView;
	private SmoothScrollRunnable mSmoothScrollRunnable;
	private FrameLayout mRefreshableViewWrapper;

	public PullToRefreshBase(Context context) {
		super(context);
		init(context, null);
	}

	public PullToRefreshBase(Context context, AttributeSet attrs) {
		super(context, attrs);
		init(context, attrs);
	}

	@TargetApi(VERSION_CODES.HONEYCOMB)
	public PullToRefreshBase(Context context, AttributeSet attrs, int defStyle) {
		super(context, attrs, defStyle);
		init(context, attrs);
	}

	private void init(Context context, AttributeSet attrs) {
		setOrientation(LinearLayout.VERTICAL);

		mTouchSlop = ViewConfiguration.get(context).getScaledTouchSlop();

		mHeaderLayout = createHeaderLoadingLayout(context, attrs);
		mFooterLayout = createFooterLoadingLayout(context, attrs);
		mRefreshableView = createRefreshableView(context, attrs);

		if (null == mRefreshableView) {
			throw new NullPointerException("Refreshable view can not be null.");
		}

		addRefreshableView(context, mRefreshableView);
		addHeaderAndFooter(context);

		// 得到Header的高度，这个高度需要用这种方式得到，在onLayout方法里面得到的高度始终是0
		getViewTreeObserver().addOnGlobalLayoutListener(new OnGlobalLayoutListener() {
			@SuppressWarnings("deprecation")
			@Override
			public void onGlobalLayout() {
				refreshLoadingViewsSize();
				getViewTreeObserver().removeGlobalOnLayoutListener(this);
			}
		});
	}

	private void refreshLoadingViewsSize() {
		// 得到header和footer的内容高度，它将会作为拖动刷新的一个临界值，如果拖动距离大于这个高度
		// 然后再松开手，就会触发刷新操作
		int headerHeight = (null != mHeaderLayout) ? mHeaderLayout.getContentSize() : 0;
		int footerHeight = (null != mFooterLayout) ? mFooterLayout.getContentSize() : 0;

		if (headerHeight < 0) {
			headerHeight = 0;
		}

		if (footerHeight < 0) {
			footerHeight = 0;
		}

		mHeaderHeight = headerHeight;
		mFooterHeight = footerHeight;

		// 这里得到Header和Footer的高度，设置的padding的top和bottom就应该是header和footer的高度
		// 因为header和footer是完全看不见的
		headerHeight = (null != mHeaderLayout) ? mHeaderLayout.getMeasuredHeight() : 0;
		footerHeight = (null != mFooterLayout) ? mFooterLayout.getMeasuredHeight() : 0;
		if (0 == footerHeight) {
			footerHeight = mFooterHeight;
		}

		int pLeft = getPaddingLeft();
		int pTop = getPaddingTop();
		int pRight = getPaddingRight();
		int pBottom = getPaddingBottom();

		pTop = -headerHeight;
		pBottom = -footerHeight;

		setPadding(pLeft, pTop, pRight, pBottom);
	}

	@Override
	protected final void onSizeChanged(int w, int h, int oldw, int oldh) {
		super.onSizeChanged(w, h, oldw, oldh);

		// We need to update the header/footer when our size changes
		refreshLoadingViewsSize();

		// 设置刷新View的大小
		refreshRefreshableViewSize(w, h);

		post(new Runnable() {
			@Override
			public void run() {
				requestLayout();
			}
		});
	}

	@Override
	public void setOrientation(int orientation) {
		if (LinearLayout.VERTICAL != orientation) {
			throw new IllegalArgumentException("This class only supports VERTICAL orientation.");
		}

		// Only support vertical orientation
		super.setOrientation(orientation);
	}

	@Override
	public final boolean onInterceptTouchEvent(MotionEvent event) {
		if (!isInterceptTouchEventEnabled()) {
			return false;
		}

		if (!isPullLoadEnabled() && !isPullRefreshEnabled()) {
			return false;
		}

		final int action = event.getAction();
		if (action == MotionEvent.ACTION_CANCEL || action == MotionEvent.ACTION_UP) {
			mIsHandledTouchEvent = false;
			return false;
		}

		if (action != MotionEvent.ACTION_DOWN && mIsHandledTouchEvent) {
			return true;
		}

		switch (action) {
		case MotionEvent.ACTION_DOWN:
			mLastMotionY = event.getY();
			mIsHandledTouchEvent = false;
			break;

		case MotionEvent.ACTION_MOVE:
			final float deltaY = event.getY() - mLastMotionY;
			final float absDiff = Math.abs(deltaY);
			// 这里有三个条件：
			// 1，位移差大于mTouchSlop，这是为了防止快速拖动引发刷新
			// 2，isPullRefreshing()，如果当前正在下拉刷新的话，是允许向上滑动，并把刷新的HeaderView挤上去
			// 3，isPullLoading()，理由与第2条相同
			if (absDiff > mTouchSlop || isPullRefreshing() || isPullLoading()) {
				mLastMotionY = event.getY();
				// 第一个显示出来，Header已经显示或拉下
				if (isPullRefreshEnabled() && isReadyForPullDown()) {
					// 1，Math.abs(getScrollY()) >
					// 0：表示当前滑动的偏移量的绝对值大于0，表示当前HeaderView滑出来了或完全
					// 不可见，存在这样一种case，当正在刷新时并且RefreshableView已经滑到顶部，向上滑动，那么我们期望的结果是
					// 依然能向上滑动，直到HeaderView完全不可见
					// 2，deltaY > 0.5f：表示下拉的值大于0.5f
					mIsHandledTouchEvent = (Math.abs(getScrollYValue()) > 0 || deltaY > 0.5f);
					// 如果截断事件，我们则仍然把这个事件交给刷新View去处理，典型的情况是让ListView/GridView将按下
					// Child的Selector隐藏
					if (mIsHandledTouchEvent) {
						mRefreshableView.onTouchEvent(event);
					}
				} else if (isPullLoadEnabled() && isReadyForPullUp()) {
					// 原理如上
					mIsHandledTouchEvent = (Math.abs(getScrollYValue()) > 0 || deltaY < -0.5f);
				}
			}
			break;

		default:
			break;
		}

		return mIsHandledTouchEvent;
	}

	@Override
	public final boolean onTouchEvent(MotionEvent ev) {
		boolean handled = false;
		switch (ev.getAction()) {
		case MotionEvent.ACTION_DOWN:
			mLastMotionY = ev.getY();
			mIsHandledTouchEvent = false;
			break;

		case MotionEvent.ACTION_MOVE:
			final float deltaY = ev.getY() - mLastMotionY;
			mLastMotionY = ev.getY();
			if (isPullRefreshEnabled() && isReadyForPullDown()) {
				pullHeaderLayout(deltaY / OFFSET_RADIO);
				handled = true;
			} else if (isPullLoadEnabled() && isReadyForPullUp()) {
				pullFooterLayout(deltaY / OFFSET_RADIO);
				handled = true;
			} else {
				mIsHandledTouchEvent = false;
			}
			break;

		case MotionEvent.ACTION_CANCEL:
		case MotionEvent.ACTION_UP:
			if (mIsHandledTouchEvent) {
				mIsHandledTouchEvent = false;
				// 当第一个显示出来时
				if (isReadyForPullDown()) {
					// 调用刷新
					if (mPullRefreshEnabled && (mPullDownState == ILoadingLayout.State.RELEASE_TO_REFRESH)) {
						startRefreshing();
						handled = true;
					}
					resetHeaderLayout();
				} else if (isReadyForPullUp()) {
					// 加载更多
					if (isPullLoadEnabled() && (mPullUpState == ILoadingLayout.State.RELEASE_TO_REFRESH)) {
						startLoading();
						handled = true;
					}
					resetFooterLayout();
				}
			}
			break;

		default:
			break;
		}

		return handled;
	}

	@Override
	public void setPullRefreshEnabled(boolean pullRefreshEnabled) {
		mPullRefreshEnabled = pullRefreshEnabled;
	}

	@Override
	public void setPullLoadEnabled(boolean pullLoadEnabled) {
		mPullLoadEnabled = pullLoadEnabled;
	}

	@Override
	public void setScrollLoadEnabled(boolean scrollLoadEnabled) {
		mScrollLoadEnabled = scrollLoadEnabled;
	}

	@Override
	public boolean isPullRefreshEnabled() {
		return mPullRefreshEnabled && (null != mHeaderLayout);
	}

	@Override
	public boolean isPullLoadEnabled() {
		return mPullLoadEnabled && (null != mFooterLayout);
	}

	@Override
	public boolean isScrollLoadEnabled() {
		return mScrollLoadEnabled;
	}

	@Override
	public void setOnRefreshListener(OnRefreshListener<T> refreshListener) {
		mRefreshListener = refreshListener;
	}

	@Override
	public void onPullDownRefreshComplete() {
		if (isPullRefreshing()) {
			mPullDownState = ILoadingLayout.State.RESET;
			onStateChanged(ILoadingLayout.State.RESET, true);

			// 回滚动有一个时间，我们在回滚完成后再设置状态为normal
			// 在将LoadingLayout的状态设置为normal之前，我们应该禁止
			// 截断Touch事件，因为设里有一个post状态，如果有post的Runnable
			// 未被执行时，用户再一次发起下拉刷新，如果正在刷新时，这个Runnable
			// 再次被执行到，那么就会把正在刷新的状态改为正常状态，这就不符合期望
			postDelayed(new Runnable() {
				@Override
				public void run() {
					setInterceptTouchEventEnabled(true);
					mHeaderLayout.setState(ILoadingLayout.State.RESET);
				}
			}, getSmoothScrollDuration());

			resetHeaderLayout();
			setInterceptTouchEventEnabled(false);
		}
	}

	@Override
	public void onPullUpRefreshComplete() {
		if (isPullLoading()) {
			mPullUpState = ILoadingLayout.State.RESET;
			onStateChanged(ILoadingLayout.State.RESET, false);

			postDelayed(new Runnable() {
				@Override
				public void run() {
					setInterceptTouchEventEnabled(true);
					mFooterLayout.setState(ILoadingLayout.State.RESET);
				}
			}, getSmoothScrollDuration());

			resetFooterLayout();
			setInterceptTouchEventEnabled(false);
		}
	}

	@Override
	public T getRefreshableView() {
		return mRefreshableView;
	}

	@Override
	public LoadingLayout getHeaderLoadingLayout() {
		return mHeaderLayout;
	}

	@Override
	public LoadingLayout getFooterLoadingLayout() {
		return mFooterLayout;
	}

	@Override
	public void setLastUpdatedLabel(CharSequence label) {
		if (null != mHeaderLayout) {
			mHeaderLayout.setLastUpdatedLabel(label);
		}

		if (null != mFooterLayout) {
			mFooterLayout.setLastUpdatedLabel(label);
		}
	}

	public void doPullRefreshing(final boolean smoothScroll, final long delayMillis) {
		postDelayed(new Runnable() {
			@Override
			public void run() {
				int newScrollValue = -mHeaderHeight;
				int duration = smoothScroll ? SCROLL_DURATION : 0;

				startRefreshing();
				smoothScrollTo(newScrollValue, duration, 0);
			}
		}, delayMillis);
	}

	protected abstract T createRefreshableView(Context context, AttributeSet attrs);

	protected abstract boolean isReadyForPullDown();

	protected abstract boolean isReadyForPullUp();

	protected LoadingLayout createHeaderLoadingLayout(Context context, AttributeSet attrs) {
		return new HeaderLoadingLayout(context);
	}

	protected LoadingLayout createFooterLoadingLayout(Context context, AttributeSet attrs) {
		return new FooterLoadingLayout(context);
	}

	protected long getSmoothScrollDuration() {
		return SCROLL_DURATION;
	}

	protected void refreshRefreshableViewSize(int width, int height) {
		if (null != mRefreshableViewWrapper) {
			LayoutParams lp = (LayoutParams) mRefreshableViewWrapper.getLayoutParams();
			if (lp.height != height) {
				lp.height = height;
				mRefreshableViewWrapper.requestLayout();
			}
		}
	}

	protected void addRefreshableView(Context context, T refreshableView) {
		int width = ViewGroup.LayoutParams.MATCH_PARENT;
		int height = ViewGroup.LayoutParams.MATCH_PARENT;

		// 创建一个包装容器
		mRefreshableViewWrapper = new FrameLayout(context);
		mRefreshableViewWrapper.addView(refreshableView, width, height);

		// 这里把Refresh view的高度设置为一个很小的值，它的高度最终会在onSizeChanged()方法中设置为MATCH_PARENT
		// 这样做的原因是，如果此是它的height是MATCH_PARENT，那么footer得到的高度就是0，所以，我们先设置高度很小
		// 我们就可以得到header和footer的正常高度，当onSizeChanged后，Refresh view的高度又会变为正常。
		height = 10;
		addView(mRefreshableViewWrapper, new LayoutParams(width, height));
	}

	protected void addHeaderAndFooter(Context context) {
		LayoutParams params = new LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT,
				ViewGroup.LayoutParams.WRAP_CONTENT);

		final LoadingLayout headerLayout = mHeaderLayout;
		final LoadingLayout footerLayout = mFooterLayout;

		if (null != headerLayout) {
			if (this == headerLayout.getParent()) {
				removeView(headerLayout);
			}

			addView(headerLayout, 0, params);
		}

		if (null != footerLayout) {
			if (this == footerLayout.getParent()) {
				removeView(footerLayout);
			}

			addView(footerLayout, -1, params);
		}
	}

	protected void pullHeaderLayout(float delta) {
		// 向上滑动，并且当前scrollY为0时，不滑动
		int oldScrollY = getScrollYValue();
		if (delta < 0 && (oldScrollY - delta) >= 0) {
			setScrollTo(0, 0);
			return;
		}

		// 向下滑动布局
		setScrollBy(0, -(int) delta);

		if (null != mHeaderLayout && 0 != mHeaderHeight) {
			float scale = Math.abs(getScrollYValue()) / (float) mHeaderHeight;
			mHeaderLayout.onPull(scale);
		}

		// 未处于刷新状态，更新箭头
		int scrollY = Math.abs(getScrollYValue());
		if (isPullRefreshEnabled() && !isPullRefreshing()) {
			if (scrollY > mHeaderHeight) {
				mPullDownState = ILoadingLayout.State.RELEASE_TO_REFRESH;
			} else {
				mPullDownState = ILoadingLayout.State.PULL_TO_REFRESH;
			}

			mHeaderLayout.setState(mPullDownState);
			onStateChanged(mPullDownState, true);
		}
	}

	protected void pullFooterLayout(float delta) {
		int oldScrollY = getScrollYValue();
		if (delta > 0 && (oldScrollY - delta) <= 0) {
			setScrollTo(0, 0);
			return;
		}

		setScrollBy(0, -(int) delta);

		if (null != mFooterLayout && 0 != mFooterHeight) {
			float scale = Math.abs(getScrollYValue()) / (float) mFooterHeight;
			mFooterLayout.onPull(scale);
		}

		int scrollY = Math.abs(getScrollYValue());
		if (isPullLoadEnabled() && !isPullLoading()) {
			if (scrollY > mFooterHeight) {
				mPullUpState = ILoadingLayout.State.RELEASE_TO_REFRESH;
			} else {
				mPullUpState = ILoadingLayout.State.PULL_TO_REFRESH;
			}

			mFooterLayout.setState(mPullUpState);
			onStateChanged(mPullUpState, false);
		}
	}

	protected void resetHeaderLayout() {
		final int scrollY = Math.abs(getScrollYValue());
		final boolean refreshing = isPullRefreshing();

		if (refreshing && scrollY <= mHeaderHeight) {
			smoothScrollTo(0);
			return;
		}

		if (refreshing) {
			smoothScrollTo(-mHeaderHeight);
		} else {
			smoothScrollTo(0);
		}
	}

	protected void resetFooterLayout() {
		int scrollY = Math.abs(getScrollYValue());
		boolean isPullLoading = isPullLoading();

		if (isPullLoading && scrollY <= mFooterHeight) {
			smoothScrollTo(0);
			return;
		}

		if (isPullLoading) {
			smoothScrollTo(mFooterHeight);
		} else {
			smoothScrollTo(0);
		}
	}

	protected boolean isPullRefreshing() {
		return (mPullDownState == ILoadingLayout.State.REFRESHING);
	}

	protected boolean isPullLoading() {
		return (mPullUpState == ILoadingLayout.State.REFRESHING);
	}

	protected void startRefreshing() {
		// 如果正在刷新
		if (isPullRefreshing()) {
			return;
		}

		mPullDownState = ILoadingLayout.State.REFRESHING;
		onStateChanged(ILoadingLayout.State.REFRESHING, true);

		if (null != mHeaderLayout) {
			mHeaderLayout.setState(ILoadingLayout.State.REFRESHING);
		}

		if (null != mRefreshListener) {
			// 因为滚动回原始位置的时间是200，我们需要等回滚完后才执行刷新回调
			postDelayed(new Runnable() {
				@Override
				public void run() {
					mRefreshListener.onPullDownToRefresh(PullToRefreshBase.this);
				}
			}, getSmoothScrollDuration());
		}
	}

	protected void startLoading() {
		// 如果正在加载
		if (isPullLoading()) {
			return;
		}

		mPullUpState = ILoadingLayout.State.REFRESHING;
		onStateChanged(ILoadingLayout.State.REFRESHING, false);

		if (null != mFooterLayout) {
			mFooterLayout.setState(ILoadingLayout.State.REFRESHING);
		}

		if (null != mRefreshListener) {
			// 因为滚动回原始位置的时间是200，我们需要等回滚完后才执行加载回调
			postDelayed(new Runnable() {
				@Override
				public void run() {
					mRefreshListener.onPullUpToRefresh(PullToRefreshBase.this);
				}
			}, getSmoothScrollDuration());
		}
	}

	protected void onStateChanged(ILoadingLayout.State state, boolean isPullDown) {

	}

	private void setScrollTo(int x, int y) {
		scrollTo(x, y);
	}

	private void setScrollBy(int x, int y) {
		scrollBy(x, y);
	}

	private int getScrollYValue() {
		return getScrollY();
	}

	private void smoothScrollTo(int newScrollValue) {
		smoothScrollTo(newScrollValue, getSmoothScrollDuration(), 0);
	}

	private void smoothScrollTo(int newScrollValue, long duration, long delayMillis) {
		if (null != mSmoothScrollRunnable) {
			mSmoothScrollRunnable.stop();
		}

		int oldScrollValue = this.getScrollYValue();
		boolean post = (oldScrollValue != newScrollValue);

		if (null == mScrollAnimationInterpolator) {
			// Default interpolator is a Decelerate Interpolator
			mScrollAnimationInterpolator = new DecelerateInterpolator();
		}

		if (post) {
			mSmoothScrollRunnable = new SmoothScrollRunnable(oldScrollValue, newScrollValue, duration);
		}

		if (post) {
			if (delayMillis > 0) {
				postDelayed(mSmoothScrollRunnable, delayMillis);
			} else {
				post(mSmoothScrollRunnable);
			}
		}
	}

	private void setInterceptTouchEventEnabled(boolean enabled) {
		mInterceptEventEnable = enabled;
	}

	private boolean isInterceptTouchEventEnabled() {
		return mInterceptEventEnable;
	}

	final class SmoothScrollRunnable implements Runnable {

		private final Interpolator mInterpolator;

		private final int mScrollToY;

		private final int mScrollFromY;

		private final long mDuration;

		private boolean mContinueRunning = true;

		private long mStartTime = -1;

		private int mCurrentY = -1;

		public SmoothScrollRunnable(int fromY, int toY, long duration) {
			mScrollFromY = fromY;
			mScrollToY = toY;
			mDuration = duration;
			mInterpolator = mScrollAnimationInterpolator;
		}

		@Override
		public void run() {

			if (mDuration <= 0) {
				setScrollTo(0, mScrollToY);
				return;
			}

			if (mStartTime == -1) {
				mStartTime = System.currentTimeMillis();
			} else {

				long normalizedTime = (1000 * (System.currentTimeMillis() - mStartTime)) / mDuration;
				normalizedTime = Math.max(Math.min(normalizedTime, 1000), 0);

				final int deltaY = Math
						.round((mScrollFromY - mScrollToY) * mInterpolator.getInterpolation(normalizedTime / 1000f));
				mCurrentY = mScrollFromY - deltaY;
				setScrollTo(0, mCurrentY);

			}

			// If we're not at the target Y, keep going...
			if (mContinueRunning && mScrollToY != mCurrentY) {
				postOnAnimation(PullToRefreshBase.this, this);
			}
		}

		public void stop() {
			mContinueRunning = false;
			removeCallbacks(this);
		}
	}

	public void postOnAnimation(View view, Runnable runnable) {
		if (VERSION.SDK_INT >= VERSION_CODES.JELLY_BEAN) {
			SDK16.postOnAnimation(view, runnable);
		} else {
			view.postDelayed(runnable, 20);
		}
	}

	@TargetApi(16)
	static class SDK16 {

		public static void postOnAnimation(View view, Runnable runnable) {
			view.postOnAnimation(runnable);
		}
	}
}
